#!/usr/bin/perl -w
##
## patchUtil.pl
##
## (c)  Copyright  IBM Corp.  2004  All Rights Reserved
##
## US Government Users Restricted Rights - Use, duplication or
## disclosure restricted by GSA ADP Schedule Contract with IBM Corp
##
## Module History:
##  -00 03/05/2004 John Thornton   LM28 initial release
##  -01 03/08/2004 John Thornton   LM29 delete support
##  -02 03/10/2004 John Thornton   LM29 "is directory" support
##
##

#
# Includes
#
use Getopt::Long;

##############################################################################
#####                          GLOBAL VARIABLES                          #####
##############################################################################

# Command Constants
$UTIL_CMD_NONE          = 0;
$UTIL_CMD_EXISTS        = 1;
$UTIL_CMD_MKDIRS        = 2;
$UTIL_CMD_DELETE        = 3;                          #-01
$UTIL_CMD_ISDIR         = 4;                          #-02

# Trace Constants
# $TRC_T = "XMCLSHLT:";
$TRC_D = "XMCLSHLD:";
$TRC_F = "XMCLSHLF:";

# Error Constants
$ERROR_ALL_NOERROR            = 0;
$ERROR_ALL_UNKNOWN_COMMAND    = 255;
                                                      #-01
$ERROR_DELETE_NOTFOUND        = 1;                    #-01
$ERROR_DELETE_ERROR           = 3;                    #-01

$ERROR_EXISTS_NOTFOUND        = 1;

$ERROR_ISDIR_NOTFOUND         = 1;                    #-02

$ERROR_MKDIRS_EXISTS          = 1;
$ERROR_MKDIRS_ISFILE          = 2;
$ERROR_MKDIRS_ERROR           = 3;

# Program Data
$me = $0;
$me =~ s/^(.*)\///;
$me =~ s/.pl$//;
$cmd = $UTIL_CMD_NONE;
$traceCmd = "actzTrace";

##############################################################################
#####                          SHARED METHODS                            #####
##############################################################################

#
#
#
sub trace
{
    my $text = shift;
    chomp($text);

    if ("actzTrace" eq "$traceCmd")
    {
       system(("$traceCmd", "$text"));
    }
    else
    {
       print(STDOUT "$text\n");
    }
}

#
# usage
#
sub usage
{
    print(<<EOM);
Usage: patchUtil [-help] command=cmd [source=sfile] [target=tfile] [...]
   cmd   an integer value that represent a command
            1 - exists()
            2 - mkdirs()
   any other arguments are optional and will vary from command to command
EOM
}

##############################################################################
#####                          UTILITY METHODS                           #####
##############################################################################

# start -01
#
#
sub doDelete
{
   trace("$TRC_D -> $me.doDelete()");
   my $file = shift;
   my $rc = $ERROR_DELETE_NOTFOUND;
   if (-e "$file")
   {
      my $error = `rm -f $target 2>&1`;
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doDelete() - rm error is \"$error\"");
         $rc = $ERROR_DELETE_ERROR
      }
   }
   trace("$TRC_D <- $me.doDelete() - rc = $rc");
   return($rc);
}
# end -01

#
#
#
sub doExists
{
   trace("$TRC_D -> $me.doExists()");
   my $file = shift;
   my $rc = $ERROR_EXISTS_NOTFOUND;
   # Does the path directory exist?                   #02
   if (-e "$file")
   {
      $rc = $ERROR_ALL_NOERROR;
   }
   trace("$TRC_D <- $me.doExists() - rc = $rc");
   return($rc);
}

# start -02
#
#
sub doIsDir
{
   trace("$TRC_D -> $me.doIsDir()");
   my $file = shift;
   my $rc = $ERROR_ISDIR_NOTFOUND;
   # Does a directory exist?
   if (-d "$file")
   {
      $rc = $ERROR_ALL_NOERROR;
   }
   trace("$TRC_D <- $me.doIsDir() - rc = $rc");
   return($rc);
}
# end -02

#
#
#
sub doMkDirs
{
   trace("$TRC_D -> $me.doMkDirs()");
   my $file = shift;
   my $rc = $ERROR_ALL_NOERROR;
   # Does it already exist?
   my $exists = (-e "$file");
   # Is it a directory?
   my $isDir = (-d "$file");
   if ($exists)
   {
      if ($isDir)
      {
         $rc = $ERROR_MKDIRS_EXISTS;
      }
      else
      {
         $rc = $ERROR_MKDIRS_ISFILE;
      }
   }
   else
   {
      my $error = `mkdir -p $target 2>&1`;
      chop $error;
      $rc = $? >> 8;
      if ($rc)
      {
         trace("$TRC_F    $me.doMkDirs() - mkdir error is \"$error\"");
         $rc = $ERROR_MKDIRS_ERROR
      }
   }
   trace("$TRC_D <- $me.doMkDirs() - rc = $rc");
   return($rc);
}

##############################################################################
#####                           SCRIPT CODE                              #####
##############################################################################

# Process the arguments
GetOptions(
      'help'            =>\$help,
      'command=i'       =>\$cmd,
#      'source:s'        =>\$source,
      'tracecommand:s'  =>\$traceCmd,
      'target:s'        =>\$target,
#      'ownerinfo:s'     =>\$ownerInfo,
#      'permissions:s'   =>\$permissions
);

trace("$TRC_F -> $me(@ARGV)");

unless($> == 0) {
    error("You must be root.");
}

if ($help)
{
   usage();
   exit;
}

my $rc = $ERROR_ALL_NOERROR;

# Route the command
if ($cmd == $UTIL_CMD_EXISTS)
{
   $rc = doExists($target);
}
elsif ($cmd == $UTIL_CMD_MKDIRS)
{
   $rc = doMkDirs($target);
}
elsif ($cmd == $UTIL_CMD_DELETE)                      #01
{                                                     #01
   $rc = doDelete($target);                           #01
}                                                     #01
elsif ($cmd == $UTIL_CMD_ISDIR)                       #02
{                                                     #02
   $rc = doIsDir($target);                            #02
}                                                     #02
else
{
   # Unknown command
   $rc = $ERROR_ALL_UNKNOWN_COMMAND;
}

trace("$TRC_F <- $me() - rc = $rc");

exit($rc);
